'use strict';

var Transform = require('stream').Transform;
var Util = require('util');

var PREFIX_LENGTH = 4;

function Decoder(schema) {
  this._schema = schema;
  this._buffer = null;
  Transform.call(this, {objectMode: true});
}

Util.inherits(Decoder, Transform);

Decoder.prototype._transform = function(chunk, encoding, callback) {
  if (!Buffer.isBuffer(chunk)) {
    return callback(new Error('chunk must be a buffer'));
  }

  // If there is buffered data
  if (this._buffer) {
    // Prepend the buffered data to the chunk
    chunk = Buffer.concat([this._buffer, chunk]);
    this._buffer = null;
  }

  // While there is still data
  while (chunk.length > 0) {
    // If the chunk is shorter than the prefix length
    if (chunk.length < PREFIX_LENGTH) {
      this._buffer = chunk;
      return callback();
    }

    // Read the length of the frame
    var frameLength = chunk.readUInt32BE(0);
    var endIndex = PREFIX_LENGTH + frameLength;

    // If the chunk does not contain the full frame
    if (chunk.length < endIndex) {
      this._buffer = chunk;
      return callback();
    }

    // Get the part of the buffer which contains the message
    var message = chunk.slice(PREFIX_LENGTH, endIndex);
    this.push(this._schema.decode(message));

    // Trim the chunk to the remaining data
    chunk = chunk.slice(endIndex);
  }

  return callback();
};

module.exports = Decoder;
